<?php

namespace App\Domain\Account\Models;

use App\Domain\Shared\Traits\UsesTenantConnection;
use Database\Factories\TransactionProjectionFactory;
use Illuminate\Database\Eloquent\Concerns\HasUuids;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * @property int $id
 * @property string $uuid
 * @property string $account_uuid
 * @property string $asset_code
 * @property int $amount
 * @property string $type
 * @property string|null $subtype
 * @property string|null $description
 * @property string|null $reference
 * @property string|null $external_reference
 * @property string $hash
 * @property array|null $metadata
 * @property string $status
 * @property string|null $related_account_uuid
 * @property string|null $transaction_group_uuid
 * @property string|null $parent_transaction_id
 * @property \Illuminate\Support\Carbon|null $cancelled_at
 * @property string|null $cancelled_by
 * @property \Illuminate\Support\Carbon|null $retried_at
 * @property string|null $retry_transaction_id
 * @property \Illuminate\Support\Carbon|null $created_at
 * @property \Illuminate\Support\Carbon|null $updated_at
 *
 * @method static \Illuminate\Database\Eloquent\Builder whereDate(string $column, string|\DateTimeInterface $value)
 * @method static \Illuminate\Database\Eloquent\Builder whereMonth(string $column, string|\DateTimeInterface $value)
 * @method static \Illuminate\Database\Eloquent\Builder where(string $column, mixed $operator = null, mixed $value = null, string $boolean = 'and')
 * @method static \Illuminate\Database\Eloquent\Builder whereIn(string $column, mixed $values, string $boolean = 'and', bool $not = false)
 * @method static \Illuminate\Database\Eloquent\Builder with(array|string $relations)
 * @method static \Illuminate\Database\Eloquent\Builder distinct(string $column = null)
 * @method static mixed sum(string $column)
 * @method static mixed avg(string $column)
 * @method static mixed max(string $column)
 * @method static int count(string $columns = '*')
 * @method static \Illuminate\Database\Eloquent\Collection get(array $columns = ['*'])
 * @method static \Illuminate\Database\Eloquent\Builder orderBy(string $column, string $direction = 'asc')
 * @method static \Illuminate\Database\Eloquent\Builder whereBetween(string $column, array $values)
 * @method static bool update(array $attributes = [])
 */
class TransactionProjection extends Model
{
    use UsesTenantConnection;
    use HasFactory;
    use HasUuids;

    protected $table = 'transaction_projections';

    protected $fillable = [
        'uuid',
        'account_uuid',
        'asset_code',
        'amount',
        'type',
        'subtype',
        'description',
        'reference',
        'external_reference',
        'hash',
        'metadata',
        'status',
        'related_account_uuid',
        'transaction_group_uuid',
        'parent_transaction_id',
        'cancelled_at',
        'cancelled_by',
        'retried_at',
        'retry_transaction_id',
    ];

    protected $casts = [
        'amount'     => 'integer',
        'metadata'   => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the columns that should receive a unique identifier.
     */
    public function uniqueIds(): array
    {
        return ['uuid'];
    }

    /**
     * Create a new factory instance for the model.
     */
    protected static function newFactory(): TransactionProjectionFactory
    {
        return TransactionProjectionFactory::new();
    }

    /**
     * Get the account that owns the transaction.
     */
    /**
     * @return BelongsTo<Account, $this>
     */
    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class, 'account_uuid', 'uuid');
    }

    /**
     * Get formatted amount (in dollars/cents format).
     */
    public function getFormattedAmountAttribute(): string
    {
        return number_format($this->amount / 100, 2);
    }

    /**
     * Scope to get transactions for a specific type.
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Scope to get completed transactions.
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Get the activity logs for this model.
     */
    /**
     * @return \Illuminate\Database\Eloquent\Relations\MorphMany
     */
    public function logs()
    {
        return $this->morphMany(\App\Domain\Activity\Models\Activity::class, 'subject');
    }
}
