<?php

namespace App\Domain\Regulatory\Services;

use App\Domain\Account\Models\Transaction;
use App\Domain\Compliance\Services\RegulatoryReportingService;
use App\Domain\Fraud\Models\FraudCase;
use App\Domain\Fraud\Models\FraudScore;
use App\Domain\Regulatory\Models\RegulatoryReport;
use App\Domain\Regulatory\Models\RegulatoryThreshold;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class EnhancedRegulatoryReportingService extends RegulatoryReportingService
{
    private ThresholdMonitoringService $thresholdService;

    private ReportGeneratorService $reportGenerator;

    private RegulatoryFilingService $filingService;

    public function __construct(
        ThresholdMonitoringService $thresholdService,
        ReportGeneratorService $reportGenerator,
        RegulatoryFilingService $filingService
    ) {
        $this->thresholdService = $thresholdService;
        $this->reportGenerator = $reportGenerator;
        $this->filingService = $filingService;
    }

    /**
     * Generate enhanced CTR with fraud detection integration.
     */
    public function generateEnhancedCTR(Carbon $date): RegulatoryReport
    {
        // First generate the basic CTR
        $filename = parent::generateCTR($date);

        // Create regulatory report record
        $report = RegulatoryReport::create(
            [
            'report_type'            => RegulatoryReport::TYPE_CTR,
            'jurisdiction'           => RegulatoryReport::JURISDICTION_US,
            'reporting_period_start' => $date->startOfDay(),
            'reporting_period_end'   => $date->endOfDay(),
            'status'                 => RegulatoryReport::STATUS_DRAFT,
            'priority'               => 4, // High priority
            'file_path'              => $filename,
            'file_format'            => RegulatoryReport::FORMAT_JSON,
            'file_size'              => Storage::size($filename),
            'file_hash'              => hash_file('sha256', Storage::path($filename)),
            'generated_at'           => now(),
            'regulation_reference'   => '31 CFR 1022.310',
            'is_mandatory'           => true,
            'due_date'               => $date->copy()->addBusinessDays(15),
            ]
        );

        // Enhance with fraud detection data
        $this->enhanceReportWithFraudData($report, $date);

        // Check regulatory thresholds
        $this->checkAndApplyThresholds($report);

        return $report;
    }

    /**
     * Generate enhanced SAR with comprehensive suspicious activity analysis.
     */
    public function generateEnhancedSAR(Carbon $startDate, Carbon $endDate): RegulatoryReport
    {
        // Get all fraud cases in the period
        $fraudCases = FraudCase::whereBetween('created_at', [$startDate, $endDate])
            ->where('priority', '>=', FraudCase::PRIORITY_HIGH)
            ->with(['fraudScore', 'fraudScore.entity'])
            ->get();

        // Get high-risk fraud scores without cases
        $highRiskScores = FraudScore::whereBetween('created_at', [$startDate, $endDate])
            ->where('risk_level', '>=', FraudScore::RISK_LEVEL_HIGH)
            ->whereDoesntHave('fraudCase')
            ->with('entity')
            ->get();

        // Combine suspicious activities
        $suspiciousActivities = $this->compileSuspiciousActivities(
            $fraudCases,
            $highRiskScores,
            $startDate,
            $endDate
        );

        // Generate report data
        $reportData = [
            'report_type'            => 'Enhanced Suspicious Activity Report (SAR)',
            'period_start'           => $startDate->toDateString(),
            'period_end'             => $endDate->toDateString(),
            'generated_at'           => now()->toISOString(),
            'total_activities'       => $suspiciousActivities->count(),
            'fraud_cases'            => $fraudCases->count(),
            'high_risk_transactions' => $highRiskScores->count(),
            'activities'             => $suspiciousActivities,
            'summary'                => $this->generateSARSummary($suspiciousActivities),
        ];

        // Save report
        $filename = "regulatory/sar/enhanced_sar_{$startDate->format('Y_m_d')}_{$endDate->format('Y_m_d')}.json";
        Storage::put($filename, json_encode($reportData, JSON_PRETTY_PRINT));

        // Create regulatory report record
        $report = RegulatoryReport::create(
            [
            'report_type'            => RegulatoryReport::TYPE_SAR,
            'jurisdiction'           => RegulatoryReport::JURISDICTION_US,
            'reporting_period_start' => $startDate,
            'reporting_period_end'   => $endDate,
            'status'                 => RegulatoryReport::STATUS_DRAFT,
            'priority'               => 5, // Critical priority
            'file_path'              => $filename,
            'file_format'            => RegulatoryReport::FORMAT_JSON,
            'file_size'              => Storage::size($filename),
            'file_hash'              => hash_file('sha256', Storage::path($filename)),
            'generated_at'           => now(),
            'regulation_reference'   => '31 CFR 1022.320',
            'is_mandatory'           => true,
            'due_date'               => now()->addBusinessDays(30),
            'report_data'            => [
                'total_activities'          => $suspiciousActivities->count(),
                'fraud_cases'               => $fraudCases->count(),
                'requires_immediate_filing' => $this->requiresImmediateFiling($suspiciousActivities),
            ],
            'record_count' => $suspiciousActivities->count(),
            ]
        );

        // Add entities and risk indicators
        foreach ($suspiciousActivities as $activity) {
            $report->addEntity(
                $activity['entity_type'],
                $activity['entity_id'],
                [
                'risk_score'    => $activity['risk_score'] ?? null,
                'fraud_case_id' => $activity['fraud_case_id'] ?? null,
                ]
            );

            if (! empty($activity['risk_indicators'])) {
                foreach ($activity['risk_indicators'] as $indicator) {
                    $report->addRiskIndicator(
                        $indicator,
                        'high',
                        [
                        'activity_id' => $activity['id'],
                        ]
                    );
                }
            }
        }

        return $report;
    }

    /**
     * Generate comprehensive AML report.
     */
    public function generateAMLReport(Carbon $month): RegulatoryReport
    {
        $startDate = $month->copy()->startOfMonth();
        $endDate = $month->copy()->endOfMonth();

        $reportData = [
            'report_type'            => 'Anti-Money Laundering (AML) Report',
            'month'                  => $month->format('F Y'),
            'generated_at'           => now()->toISOString(),
            'compliance_metrics'     => $this->getAMLComplianceMetrics($startDate, $endDate),
            'risk_assessment'        => $this->performAMLRiskAssessment($startDate, $endDate),
            'transaction_monitoring' => $this->getTransactionMonitoringResults($startDate, $endDate),
            'customer_risk_ratings'  => $this->getCustomerRiskRatings(),
            'sanctions_screening'    => $this->getSanctionsScreeningResults($startDate, $endDate),
            'training_compliance'    => $this->getAMLTrainingCompliance(),
            'policy_updates'         => $this->getAMLPolicyUpdates($month),
        ];

        // Save report
        $filename = "regulatory/aml/report_{$month->format('Y_m')}.json";
        Storage::put($filename, json_encode($reportData, JSON_PRETTY_PRINT));

        // Create regulatory report record
        $report = RegulatoryReport::create(
            [
            'report_type'            => RegulatoryReport::TYPE_AML,
            'jurisdiction'           => RegulatoryReport::JURISDICTION_US,
            'reporting_period_start' => $startDate,
            'reporting_period_end'   => $endDate,
            'status'                 => RegulatoryReport::STATUS_DRAFT,
            'priority'               => 3,
            'file_path'              => $filename,
            'file_format'            => RegulatoryReport::FORMAT_JSON,
            'file_size'              => Storage::size($filename),
            'file_hash'              => hash_file('sha256', Storage::path($filename)),
            'generated_at'           => now(),
            'regulation_reference'   => 'BSA/AML',
            'is_mandatory'           => true,
            'due_date'               => $endDate->copy()->addBusinessDays(10),
            'report_data'            => [
                'total_transactions_monitored' => $reportData['transaction_monitoring']['total_monitored'],
                'suspicious_transactions'      => $reportData['transaction_monitoring']['suspicious_count'],
                'high_risk_customers'          => $reportData['customer_risk_ratings']['high_risk_count'],
                'sanctions_hits'               => $reportData['sanctions_screening']['total_hits'],
            ],
            ]
        );

        return $report;
    }

    /**
     * Generate OFAC screening report.
     */
    public function generateOFACReport(Carbon $date): RegulatoryReport
    {
        $reportData = [
            'report_type'          => 'OFAC Screening Report',
            'date'                 => $date->toDateString(),
            'generated_at'         => now()->toISOString(),
            'screening_results'    => $this->performOFACScreening($date),
            'blocked_transactions' => $this->getBlockedTransactions($date),
            'false_positives'      => $this->getOFACFalsePositives($date),
            'remediation_actions'  => $this->getOFACRemediationActions($date),
        ];

        // Save report
        $filename = "regulatory/ofac/report_{$date->format('Y_m_d')}.json";
        Storage::put($filename, json_encode($reportData, JSON_PRETTY_PRINT));

        // Create regulatory report record
        $report = RegulatoryReport::create(
            [
            'report_type'            => RegulatoryReport::TYPE_OFAC,
            'jurisdiction'           => RegulatoryReport::JURISDICTION_US,
            'reporting_period_start' => $date->startOfDay(),
            'reporting_period_end'   => $date->endOfDay(),
            'status'                 => RegulatoryReport::STATUS_DRAFT,
            'priority'               => 5, // Critical for OFAC
            'file_path'              => $filename,
            'file_format'            => RegulatoryReport::FORMAT_JSON,
            'file_size'              => Storage::size($filename),
            'file_hash'              => hash_file('sha256', Storage::path($filename)),
            'generated_at'           => now(),
            'regulation_reference'   => 'OFAC Sanctions',
            'is_mandatory'           => true,
            'due_date'               => $date->copy()->addBusinessDays(1), // Immediate reporting required
            'report_data'            => [
                'total_screened'            => $reportData['screening_results']['total_screened'],
                'matches_found'             => $reportData['screening_results']['matches_found'],
                'blocked_count'             => count($reportData['blocked_transactions']),
                'requires_immediate_action' => $reportData['screening_results']['matches_found'] > 0,
            ],
            ]
        );

        return $report;
    }

    /**
     * Generate BSA compliance report.
     */
    public function generateBSAReport(Carbon $quarter): RegulatoryReport
    {
        $startDate = $quarter->copy()->firstOfQuarter();
        $endDate = $quarter->copy()->lastOfQuarter();

        $reportData = [
            'report_type'             => 'Bank Secrecy Act (BSA) Compliance Report',
            'quarter'                 => $quarter->quarter . ' ' . $quarter->year,
            'generated_at'            => now()->toISOString(),
            'ctr_filings'             => $this->getBSACTRFilings($startDate, $endDate),
            'sar_filings'             => $this->getBSASARFilings($startDate, $endDate),
            'customer_identification' => $this->getBSACustomerIdentification($startDate, $endDate),
            'recordkeeping'           => $this->getBSARecordkeeping($startDate, $endDate),
            'compliance_testing'      => $this->getBSAComplianceTesting($quarter),
            'risk_assessment'         => $this->getBSARiskAssessment($quarter),
        ];

        // Save report
        $filename = "regulatory/bsa/report_Q{$quarter->quarter}_{$quarter->year}.json";
        Storage::put($filename, json_encode($reportData, JSON_PRETTY_PRINT));

        // Create regulatory report record
        $report = RegulatoryReport::create(
            [
            'report_type'            => RegulatoryReport::TYPE_BSA,
            'jurisdiction'           => RegulatoryReport::JURISDICTION_US,
            'reporting_period_start' => $startDate,
            'reporting_period_end'   => $endDate,
            'status'                 => RegulatoryReport::STATUS_DRAFT,
            'priority'               => 4,
            'file_path'              => $filename,
            'file_format'            => RegulatoryReport::FORMAT_JSON,
            'file_size'              => Storage::size($filename),
            'file_hash'              => hash_file('sha256', Storage::path($filename)),
            'generated_at'           => now(),
            'regulation_reference'   => 'Bank Secrecy Act',
            'is_mandatory'           => true,
            'due_date'               => $endDate->copy()->addBusinessDays(30),
            'report_data'            => [
                'ctr_count'        => $reportData['ctr_filings']['total_filed'],
                'sar_count'        => $reportData['sar_filings']['total_filed'],
                'compliance_score' => $reportData['compliance_testing']['overall_score'],
                'risk_rating'      => $reportData['risk_assessment']['overall_rating'],
            ],
            ]
        );

        return $report;
    }

    /**
     * Enhance report with fraud detection data.
     */
    protected function enhanceReportWithFraudData(RegulatoryReport $report, Carbon $date): void
    {
        // Get fraud scores for the date
        $fraudScores = FraudScore::whereDate('created_at', $date)
            ->where('risk_level', '>=', FraudScore::RISK_LEVEL_HIGH)
            ->with('entity')
            ->get();

        $enhancedData = $report->report_data ?? [];
        $enhancedData['fraud_analysis'] = [
            'high_risk_transactions' => $fraudScores->count(),
            'blocked_transactions'   => $fraudScores->where('decision', FraudScore::DECISION_BLOCK)->count(),
            'fraud_indicators'       => $this->extractFraudIndicators($fraudScores),
        ];

        $report->update(['report_data' => $enhancedData]);

        // Add risk indicators
        foreach ($fraudScores as $score) {
            if ($score->risk_level === FraudScore::RISK_LEVEL_VERY_HIGH) {
                $report->addRiskIndicator(
                    'very_high_fraud_risk',
                    'critical',
                    [
                    'fraud_score_id' => $score->id,
                    'risk_score'     => $score->total_score,
                    ]
                );
            }
        }
    }

    /**
     * Check and apply regulatory thresholds.
     */
    protected function checkAndApplyThresholds(RegulatoryReport $report): void
    {
        $thresholds = RegulatoryThreshold::active()
            ->byReportType($report->report_type)
            ->byJurisdiction($report->jurisdiction)
            ->get();

        foreach ($thresholds as $threshold) {
            $context = $this->buildThresholdContext($report);

            if ($threshold->evaluate($context)) {
                $threshold->recordTrigger();

                // Apply threshold actions
                foreach ($threshold->actions as $action) {
                    $this->applyThresholdAction($report, $threshold, $action);
                }

                // Add to report
                $report->addRiskIndicator(
                    "threshold_triggered:{$threshold->threshold_code}",
                    'high',
                    [
                    'threshold_name' => $threshold->name,
                    'triggered_at'   => now()->toIso8601String(),
                    ]
                );
            }
        }
    }

    /**
     * Build context for threshold evaluation.
     */
    protected function buildThresholdContext(RegulatoryReport $report): array
    {
        return [
            'report_type'           => $report->report_type,
            'jurisdiction'          => $report->jurisdiction,
            'record_count'          => $report->record_count,
            'total_amount'          => $report->total_amount,
            'risk_indicators_count' => count($report->risk_indicators ?? []),
            'priority'              => $report->priority,
            'is_overdue'            => $report->is_overdue,
        ];
    }

    /**
     * Apply threshold action.
     */
    protected function applyThresholdAction(RegulatoryReport $report, RegulatoryThreshold $threshold, string $action): void
    {
        switch ($action) {
            case RegulatoryThreshold::ACTION_REPORT:
                if ($threshold->shouldAutoReport()) {
                    $report->update(['status' => RegulatoryReport::STATUS_PENDING_REVIEW]);
                }
                break;

            case RegulatoryThreshold::ACTION_FLAG:
                $report->update(['priority' => min(5, $report->priority + 1)]);
                break;

            case RegulatoryThreshold::ACTION_NOTIFY:
                $this->sendThresholdNotification($report, $threshold);
                break;

            case RegulatoryThreshold::ACTION_REVIEW:
                $report->update(['requires_correction' => true]);
                break;
        }
    }

    /**
     * Compile suspicious activities from various sources.
     */
    protected function compileSuspiciousActivities(
        Collection $fraudCases,
        Collection $highRiskScores,
        Carbon $startDate,
        Carbon $endDate
    ): Collection {
        $activities = collect();

        // Add fraud cases
        foreach ($fraudCases as $case) {
            $activities->push(
                [
                'id'              => $case->id,
                'type'            => 'fraud_case',
                'entity_type'     => $case->entity_type,
                'entity_id'       => $case->entity_id,
                'risk_score'      => $case->total_score,
                'fraud_case_id'   => $case->id,
                'risk_indicators' => $case->risk_factors,
                'detected_at'     => $case->created_at->toIso8601String(),
                'status'          => $case->status,
                'priority'        => $case->priority,
                ]
            );
        }

        // Add high-risk transactions
        foreach ($highRiskScores as $score) {
            $activities->push(
                [
                'id'              => $score->id,
                'type'            => 'high_risk_transaction',
                'entity_type'     => $score->entity_type,
                'entity_id'       => $score->entity_id,
                'risk_score'      => $score->total_score,
                'risk_indicators' => $score->triggered_rules,
                'detected_at'     => $score->created_at->toIso8601String(),
                'decision'        => $score->decision,
                ]
            );
        }

        // Add pattern-based suspicious activities
        $patterns = $this->detectSuspiciousPatterns($startDate, $endDate);
        foreach ($patterns as $pattern) {
            $activities->push($pattern);
        }

        return $activities->sortByDesc('risk_score');
    }

    /**
     * Generate SAR summary.
     */
    protected function generateSARSummary(Collection $activities): array
    {
        return [
            'total_activities'  => $activities->count(),
            'by_type'           => $activities->groupBy('type')->map->count(),
            'risk_distribution' => [
                'critical' => $activities->where('risk_score', '>=', 90)->count(),
                'high'     => $activities->whereBetween('risk_score', [70, 89])->count(),
                'medium'   => $activities->whereBetween('risk_score', [50, 69])->count(),
                'low'      => $activities->where('risk_score', '<', 50)->count(),
            ],
            'requires_immediate_filing' => $activities->where('priority', '>=', 4)->count() > 0,
            'estimated_loss'            => $activities->sum('loss_amount'),
        ];
    }

    /**
     * Check if activities require immediate filing.
     */
    protected function requiresImmediateFiling(Collection $activities): bool
    {
        return $activities->contains(
            function ($activity) {
                return ($activity['risk_score'] ?? 0) >= 90 ||
                   ($activity['priority'] ?? 0) >= 5 ||
                   in_array('immediate_threat', $activity['risk_indicators'] ?? []);
            }
        );
    }

    /**
     * Extract fraud indicators from fraud scores.
     */
    protected function extractFraudIndicators(Collection $fraudScores): array
    {
        $indicators = [];

        foreach ($fraudScores as $score) {
            foreach ($score->triggered_rules ?? [] as $rule) {
                $indicators[$rule] = ($indicators[$rule] ?? 0) + 1;
            }
        }

        return $indicators;
    }

    /**
     * Send threshold notification.
     */
    protected function sendThresholdNotification(RegulatoryReport $report, RegulatoryThreshold $threshold): void
    {
        Log::warning(
            'Regulatory threshold triggered',
            [
            'report_id'      => $report->report_id,
            'threshold_code' => $threshold->threshold_code,
            'threshold_name' => $threshold->name,
            ]
        );

        // In production, send actual notifications (email, SMS, etc.)
    }

    // Additional helper methods for various report types...

    protected function getAMLComplianceMetrics(Carbon $startDate, Carbon $endDate): array
    {
        return [
            'transactions_monitored' => Transaction::whereBetween('created_at', [$startDate, $endDate])->count(),
            'alerts_generated'       => FraudScore::whereBetween('created_at', [$startDate, $endDate])
                ->where('risk_level', '>=', FraudScore::RISK_LEVEL_MEDIUM)
                ->count(),
            'cases_investigated' => FraudCase::whereBetween('created_at', [$startDate, $endDate])->count(),
            'sars_filed'         => RegulatoryReport::byType(RegulatoryReport::TYPE_SAR)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count(),
        ];
    }

    protected function performAMLRiskAssessment(Carbon $startDate, Carbon $endDate): array
    {
        return [
            'overall_risk' => 'medium',
            'risk_factors' => [
                'customer_risk'    => $this->assessCustomerRisk(),
                'product_risk'     => $this->assessProductRisk(),
                'geographic_risk'  => $this->assessGeographicRisk(),
                'transaction_risk' => $this->assessTransactionRisk($startDate, $endDate),
            ],
        ];
    }

    protected function getTransactionMonitoringResults(Carbon $startDate, Carbon $endDate): array
    {
        $fraudScores = FraudScore::whereBetween('created_at', [$startDate, $endDate])->get();

        return [
            'total_monitored'     => $fraudScores->count(),
            'suspicious_count'    => $fraudScores->where('risk_level', '>=', FraudScore::RISK_LEVEL_HIGH)->count(),
            'blocked_count'       => $fraudScores->where('decision', FraudScore::DECISION_BLOCK)->count(),
            'false_positive_rate' => $this->calculateFalsePositiveRate($fraudScores),
        ];
    }

    protected function getCustomerRiskRatings(): array
    {
        $users = User::all();

        return [
            'total_customers'   => $users->count(),
            'high_risk_count'   => $users->where('risk_rating', 'high')->count(),
            'medium_risk_count' => $users->where('risk_rating', 'medium')->count(),
            'low_risk_count'    => $users->where('risk_rating', 'low')->count(),
            'pep_count'         => $users->where('pep_status', true)->count(),
        ];
    }

    // Stub methods for various assessments - implement based on business logic
    protected function assessCustomerRisk(): string
    {
        return 'medium';
    }

    protected function assessProductRisk(): string
    {
        return 'low';
    }

    protected function assessGeographicRisk(): string
    {
        return 'medium';
    }

    protected function assessTransactionRisk($start, $end): string
    {
        return 'medium';
    }

    protected function calculateFalsePositiveRate($scores): float
    {
        return 12.5;
    }

    protected function getSanctionsScreeningResults($start, $end): array
    {
        return ['total_hits' => 0];
    }

    protected function getAMLTrainingCompliance(): array
    {
        return ['compliance_rate' => 95.0];
    }

    protected function getAMLPolicyUpdates($month): array
    {
        return ['updates' => []];
    }

    protected function performOFACScreening($date): array
    {
        return ['total_screened' => 0, 'matches_found' => 0];
    }

    protected function getBlockedTransactions($date): array
    {
        return [];
    }

    protected function getOFACFalsePositives($date): array
    {
        return [];
    }

    protected function getOFACRemediationActions($date): array
    {
        return [];
    }

    protected function getBSACTRFilings($start, $end): array
    {
        return ['total_filed' => 0];
    }

    protected function getBSASARFilings($start, $end): array
    {
        return ['total_filed' => 0];
    }

    protected function getBSACustomerIdentification($start, $end): array
    {
        return ['compliance_rate' => 98.0];
    }

    protected function getBSARecordkeeping($start, $end): array
    {
        return ['compliance_rate' => 99.0];
    }

    protected function getBSAComplianceTesting($quarter): array
    {
        return ['overall_score' => 95.0];
    }

    protected function getBSARiskAssessment($quarter): array
    {
        return ['overall_rating' => 'low'];
    }
}
