<?php

declare(strict_types=1);

namespace App\Http\Controllers\Api;

use App\Domain\Account\DataObjects\AccountUuid;
use App\Domain\Account\DataObjects\Money;
use App\Domain\Account\Models\Account;
use App\Domain\Account\Services\AccountService;
use App\Domain\Account\Services\Cache\AccountCacheService;
use App\Domain\Account\Workflows\CreateAccountWorkflow;
use App\Domain\Account\Workflows\DepositAccountWorkflow;
use App\Domain\Account\Workflows\DestroyAccountWorkflow;
use App\Domain\Account\Workflows\FreezeAccountWorkflow;
use App\Domain\Account\Workflows\UnfreezeAccountWorkflow;
use App\Http\Controllers\Controller;
use App\Rules\NoControlCharacters;
use App\Rules\NoSqlInjection;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use OpenApi\Attributes as OA;
use Workflow\WorkflowStub;

class AccountController extends Controller
{
    public function __construct(
        private readonly AccountService $accountService,
        private readonly AccountCacheService $accountCache
    ) {
    }

        #[OA\Get(
            path: '/api/accounts',
            operationId: 'listAccounts',
            tags: ['Accounts'],
            summary: 'List accounts',
            description: 'Retrieves a list of accounts for the authenticated user',
            security: [['sanctum' => []]]
        )]
    #[OA\Response(
        response: 200,
        description: 'List of accounts',
        content: new OA\JsonContent(properties: [
        new OA\Property(property: 'data', type: 'array', items: new OA\Items(ref: '#/components/schemas/Account')),
        ])
    )]
    public function index(Request $request): JsonResponse
    {
        // Get the authenticated user
        $user = $request->user();

        // Retrieve accounts for the authenticated user
        $accounts = Account::where('user_uuid', $user->uuid)
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json(
            [
                'data' => $accounts->map(
                    function ($account) {
                        return [
                            'uuid'       => $account->uuid,
                            'user_uuid'  => $account->user_uuid,
                            'name'       => $account->name,
                            'balance'    => $account->balance,
                            'frozen'     => $account->frozen ?? false,
                            'created_at' => $account->created_at,
                            'updated_at' => $account->updated_at,
                        ];
                    }
                ),
            ]
        );
    }

        #[OA\Post(
            path: '/api/accounts',
            operationId: 'createAccount',
            tags: ['Accounts'],
            summary: 'Create a new account',
            description: 'Creates a new bank account for a user with an optional initial balance',
            security: [['sanctum' => []]],
            requestBody: new OA\RequestBody(required: true, content: new OA\JsonContent(required: ['user_uuid', 'name'], properties: [
        new OA\Property(property: 'user_uuid', type: 'string', format: 'uuid', example: '660e8400-e29b-41d4-a716-446655440000'),
        new OA\Property(property: 'name', type: 'string', example: 'Savings Account', maxLength: 255),
        new OA\Property(property: 'initial_balance', type: 'integer', example: 10000, minimum: 0, description: 'Initial balance in cents'),
        ]))
        )]
    #[OA\Response(
        response: 201,
        description: 'Account created successfully',
        content: new OA\JsonContent(properties: [
        new OA\Property(property: 'data', ref: '#/components/schemas/Account'),
        new OA\Property(property: 'message', type: 'string', example: 'Account created successfully'),
        ])
    )]
    #[OA\Response(
        response: 422,
        description: 'Validation error',
        content: new OA\JsonContent(ref: '#/components/schemas/Error')
    )]
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate(
            [
                'name'            => ['required', 'string', 'max:255', new NoControlCharacters(), new NoSqlInjection()],
                'initial_balance' => 'sometimes|integer|min:0',
            ]
        );

        // Sanitize the account name to prevent XSS
        $sanitizedName = strip_tags($validated['name']);
        $sanitizedName = htmlspecialchars($sanitizedName, ENT_QUOTES, 'UTF-8');
        // Remove dangerous protocols
        $sanitizedName = preg_replace('/javascript:/i', '', $sanitizedName);
        $sanitizedName = preg_replace('/data:/i', '', $sanitizedName);
        $sanitizedName = preg_replace('/vbscript:/i', '', $sanitizedName);
        $sanitizedName = trim($sanitizedName);

        // Generate a UUID for the new account
        $accountUuid = Str::uuid()->toString();

        // Create the Account data object with the UUID
        // Always use the authenticated user's UUID, never from request
        $accountData = new \App\Domain\Account\DataObjects\Account(
            uuid: $accountUuid,
            name: $sanitizedName,
            userUuid: $request->user()->uuid
        );

        $workflow = WorkflowStub::make(CreateAccountWorkflow::class);
        $workflow->start($accountData);

        // If initial balance is provided, make a deposit
        if (isset($validated['initial_balance']) && $validated['initial_balance'] > 0) {
            $depositWorkflow = WorkflowStub::make(DepositAccountWorkflow::class);
            $depositWorkflow->start(
                new AccountUuid($accountUuid),
                new Money($validated['initial_balance'])
            );
        }

        // Wait a moment for the projector to create the account record
        $account = Account::where('uuid', $accountUuid)->first();

        // In test mode, the account might not exist yet, so create it
        if (! $account) {
            $account = Account::create(
                [
                    'uuid'      => $accountUuid,
                    'user_uuid' => $request->user()->uuid,
                    'name'      => $sanitizedName,
                    'balance'   => $validated['initial_balance'] ?? 0,
                ]
            );
        }

        return response()->json(
            [
                'data' => [
                    'uuid'       => $account->uuid,
                    'user_uuid'  => $account->user_uuid,
                    'name'       => $account->name,
                    'balance'    => $account->balance,
                    'frozen'     => $account->frozen ?? false,
                    'created_at' => $account->created_at,
                ],
                'message' => 'Account created successfully',
            ],
            201
        );
    }

        #[OA\Get(
            path: '/api/accounts/{uuid}',
            operationId: 'getAccount',
            tags: ['Accounts'],
            summary: 'Get account details',
            description: 'Retrieves detailed information about a specific account',
            security: [['sanctum' => []]],
            parameters: [
        new OA\Parameter(name: 'uuid', in: 'path', description: 'Account UUID', required: true, schema: new OA\Schema(type: 'string', format: 'uuid')),
        ]
        )]
    #[OA\Response(
        response: 200,
        description: 'Account details',
        content: new OA\JsonContent(properties: [
        new OA\Property(property: 'data', ref: '#/components/schemas/Account'),
        ])
    )]
    #[OA\Response(
        response: 404,
        description: 'Account not found',
        content: new OA\JsonContent(ref: '#/components/schemas/Error')
    )]
    public function show(Request $request, string $uuid): JsonResponse
    {
        // Try to get from cache first
        $account = $this->accountCache->get($uuid);

        if (! $account) {
            abort(404, 'Account not found');
        }

        // Check authorization - user must own the account
        if ($account->user_uuid !== $request->user()->uuid) {
            abort(403, 'Forbidden');
        }

        return response()->json(
            [
                'data' => [
                    'uuid'       => $account->uuid,
                    'user_uuid'  => $account->user_uuid,
                    'name'       => $account->name,
                    'balance'    => $account->balance,
                    'frozen'     => $account->frozen ?? false,
                    'created_at' => $account->created_at,
                    'updated_at' => $account->updated_at,
                ],
            ]
        );
    }

        #[OA\Delete(
            path: '/api/accounts/{uuid}',
            operationId: 'deleteAccount',
            tags: ['Accounts'],
            summary: 'Delete an account',
            description: 'Deletes an account. Account must have zero balance and not be frozen.',
            security: [['sanctum' => []]],
            parameters: [
        new OA\Parameter(name: 'uuid', in: 'path', description: 'Account UUID', required: true, schema: new OA\Schema(type: 'string', format: 'uuid')),
        ]
        )]
    #[OA\Response(
        response: 200,
        description: 'Account deletion initiated',
        content: new OA\JsonContent(properties: [
        new OA\Property(property: 'message', type: 'string', example: 'Account deletion initiated'),
        ])
    )]
    #[OA\Response(
        response: 422,
        description: 'Cannot delete account',
        content: new OA\JsonContent(ref: '#/components/schemas/Error')
    )]
    #[OA\Response(
        response: 404,
        description: 'Account not found',
        content: new OA\JsonContent(ref: '#/components/schemas/Error')
    )]
    public function destroy(Request $request, string $uuid): JsonResponse
    {
        $account = Account::where('uuid', $uuid)->firstOrFail();

        // Check authorization - user must own the account
        if ($account->user_uuid !== $request->user()->uuid) {
            abort(403, 'Forbidden');
        }

        // Check if account has any positive balance in any asset
        $hasPositiveBalance = $account->balances()
            ->where('balance', '>', 0)
            ->exists();

        if ($hasPositiveBalance) {
            return response()->json(
                [
                    'message' => 'Cannot delete account with positive balance',
                    'error'   => 'ACCOUNT_HAS_BALANCE',
                ],
                422
            );
        }

        if ($account->frozen) {
            return response()->json(
                [
                    'message' => 'Cannot delete frozen account',
                    'error'   => 'ACCOUNT_FROZEN',
                ],
                422
            );
        }

        $accountUuid = new AccountUuid($uuid);

        $workflow = WorkflowStub::make(DestroyAccountWorkflow::class);
        $workflow->start($accountUuid);

        return response()->json(
            [
                'message' => 'Account deletion initiated',
            ]
        );
    }

        #[OA\Post(
            path: '/api/accounts/{uuid}/freeze',
            operationId: 'freezeAccount',
            tags: ['Accounts'],
            summary: 'Freeze an account',
            description: 'Freezes an account to prevent any transactions',
            security: [['sanctum' => []]],
            parameters: [
        new OA\Parameter(name: 'uuid', in: 'path', description: 'Account UUID', required: true, schema: new OA\Schema(type: 'string', format: 'uuid')),
        ],
            requestBody: new OA\RequestBody(required: true, content: new OA\JsonContent(required: ['reason'], properties: [
        new OA\Property(property: 'reason', type: 'string', example: 'Suspicious activity detected', maxLength: 255),
        new OA\Property(property: 'authorized_by', type: 'string', example: 'admin@example.com', maxLength: 255),
        ]))
        )]
    #[OA\Response(
        response: 200,
        description: 'Account frozen successfully',
        content: new OA\JsonContent(properties: [
        new OA\Property(property: 'message', type: 'string', example: 'Account frozen successfully'),
        ])
    )]
    #[OA\Response(
        response: 422,
        description: 'Account already frozen',
        content: new OA\JsonContent(ref: '#/components/schemas/Error')
    )]
    public function freeze(Request $request, string $uuid): JsonResponse
    {
        $validated = $request->validate(
            [
                'reason'        => 'required|string|max:255',
                'authorized_by' => 'sometimes|string|max:255',
            ]
        );

        $account = Account::where('uuid', $uuid)->firstOrFail();

        // Check authorization - user must own the account OR be an admin
        // Use Spatie role check for admin, not tokenCan which is for Sanctum scopes
        $isAdmin = $request->user()->hasRole(['admin', 'super_admin', 'bank_admin']);
        if (! $isAdmin && $account->user_uuid !== $request->user()->uuid) {
            abort(403, 'Forbidden');
        }

        if ($account->frozen) {
            return response()->json(
                [
                    'message' => 'Account is already frozen',
                    'error'   => 'ACCOUNT_ALREADY_FROZEN',
                ],
                422
            );
        }

        $accountUuid = new AccountUuid($uuid);

        $workflow = WorkflowStub::make(FreezeAccountWorkflow::class);
        $workflow->start(
            $accountUuid,
            $validated['reason'],
            $validated['authorized_by'] ?? null
        );

        return response()->json(
            [
                'message' => 'Account frozen successfully',
            ]
        );
    }

        #[OA\Post(
            path: '/api/accounts/{uuid}/unfreeze',
            operationId: 'unfreezeAccount',
            tags: ['Accounts'],
            summary: 'Unfreeze an account',
            description: 'Unfreezes a previously frozen account',
            security: [['sanctum' => []]],
            parameters: [
        new OA\Parameter(name: 'uuid', in: 'path', description: 'Account UUID', required: true, schema: new OA\Schema(type: 'string', format: 'uuid')),
        ],
            requestBody: new OA\RequestBody(required: true, content: new OA\JsonContent(required: ['reason'], properties: [
        new OA\Property(property: 'reason', type: 'string', example: 'Investigation completed', maxLength: 255),
        new OA\Property(property: 'authorized_by', type: 'string', example: 'admin@example.com', maxLength: 255),
        ]))
        )]
    #[OA\Response(
        response: 200,
        description: 'Account unfrozen successfully',
        content: new OA\JsonContent(properties: [
        new OA\Property(property: 'message', type: 'string', example: 'Account unfrozen successfully'),
        ])
    )]
    #[OA\Response(
        response: 422,
        description: 'Account not frozen',
        content: new OA\JsonContent(ref: '#/components/schemas/Error')
    )]
    public function unfreeze(Request $request, string $uuid): JsonResponse
    {
        $validated = $request->validate(
            [
                'reason'        => 'required|string|max:255',
                'authorized_by' => 'sometimes|string|max:255',
            ]
        );

        $account = Account::where('uuid', $uuid)->firstOrFail();

        // Check authorization - user must own the account OR be an admin
        $isAdmin = $request->user()->hasRole(['admin', 'super_admin', 'bank_admin']);
        if (! $isAdmin && $account->user_uuid !== $request->user()->uuid) {
            abort(403, 'Forbidden');
        }

        if (! $account->frozen) {
            return response()->json(
                [
                    'message' => 'Account is not frozen',
                    'error'   => 'ACCOUNT_NOT_FROZEN',
                ],
                422
            );
        }

        $accountUuid = new AccountUuid($uuid);

        $workflow = WorkflowStub::make(UnfreezeAccountWorkflow::class);
        $workflow->start(
            $accountUuid,
            $validated['reason'],
            $validated['authorized_by'] ?? null
        );

        return response()->json(
            [
                'message' => 'Account unfrozen successfully',
            ]
        );
    }
}
