# Backend Handover v5.7.0 — Items Requiring Backend Work

**Date**: 2026-02-28
**From**: Mobile team
**To**: Backend dev
**Context**: Mobile is aligning all API paths to match existing backend routes (23 path/method fixes on our side). This document covers items that **only the backend can fix** — missing endpoints, stub implementations, missing response fields, and infrastructure setup.

---

## P0 — Missing Endpoints (no route or controller exists)

### 1. `POST /api/v1/wallet/recovery-shard-backup`

**Why**: The onboarding flow's recovery setup screen currently fakes the cloud backup with a `setTimeout`. Without this endpoint, users have no actual recovery mechanism if they lose their device.

**Expected request**:
```json
{
  "encrypted_shard": "<base64-encoded encrypted recovery shard>",
  "backup_provider": "icloud" | "gdrive",
  "device_id": "<device identifier>"
}
```

**Expected response**:
```json
{
  "success": true,
  "data": {
    "backup_id": "bkp_abc123",
    "provider": "icloud",
    "created_at": "2026-02-28T12:00:00Z"
  }
}
```

**Notes**:
- The shard is already encrypted client-side before sending — backend should store as opaque blob
- Consider adding `GET /api/v1/wallet/recovery-shard-backup` for restore flow
- Consider adding `DELETE` for shard revocation

---

### 2. `POST /broadcasting/auth` — WebSocket Channel Authorization

**Why**: The mobile app uses Pusher/Soketi for real-time updates (privacy proof status, payment confirmations, TrustCert updates). Channel subscriptions fail silently without this endpoint.

**Fix**: Add `->withBroadcasting(base_path('routes/channels.php'))` to `bootstrap/app.php`.

`channels.php` already defines the channels:
- `private-privacy.{userId}`
- `private-commerce.{merchantId}`
- `private-trustcert.{userId}`
- `private-user.{userId}`

The endpoint is auto-generated by Laravel Broadcasting — no controller code needed, just the registration call.

**Verify**: After adding, confirm `POST /broadcasting/auth` returns 200 with a valid Pusher auth signature for authenticated users.

---

### 3. `GET /api/v1/privacy/transaction-calldata/{txHash}`

**Why**: Mobile's privacy service calls this to fetch RAILGUN transaction calldata for proof verification. Currently the mobile mock returns a placeholder.

**Expected response**:
```json
{
  "data": {
    "tx_hash": "0xabc...",
    "calldata": "0x...",
    "block_number": 12345678,
    "timestamp": "2026-02-28T12:00:00Z"
  }
}
```

**Priority**: Lower than #1 and #2. Privacy proof verification can be deferred to a later release if needed.

---

## P1 — Missing Response Field

### 4. `GET /api/v1/wallet/recent-recipients` — add `name` field

**Current response**:
```json
{
  "data": [
    {
      "address": "0x1234...",
      "network": "polygon",
      "token": "USDC",
      "last_sent_at": "2026-02-27T15:30:00Z"
    }
  ]
}
```

**Expected response** (add `name`):
```json
{
  "data": [
    {
      "address": "0x1234...",
      "name": "Alice Johnson",
      "network": "polygon",
      "token": "USDC",
      "last_sent_at": "2026-02-27T15:30:00Z"
    }
  ]
}
```

**Where to get `name`**: From the user's contacts/address book if you have one, or from the `users` table if the recipient is a registered user. Return `null` if unknown — mobile handles the fallback (shows "Unknown" with initials derived from address hex).

**Impact if skipped**: Users see "Unknown" for all recent recipients. Functional but poor UX.

---

## P2 — Stub/Demo Implementations (return fake data)

These endpoints exist and respond, but return hardcoded or empty data. They work for demo purposes but will confuse real users.

### 5. `GET /api/v1/wallet/tokens` — hardcoded token list

**Current**: Returns a static array of `[USDC, USDT, WETH, WBTC]` regardless of chain or user.

**Expected**: Query supported tokens per network from DB or config. Should respect the `chain_id` query parameter.

---

### 6. `GET /api/v1/cards` — always returns `[]`

**Current**: Controller comment says "In production, query from database".

**Expected**: Return user's virtual cards from DB. The `CardProvisioningService@createCard` already creates cards — this endpoint needs to read them back.

---

### 7. `GET /api/v1/cards/{id}/transactions` — deterministic demo data

**Current**: Generates fake transactions seeded by `cardId`. Same transactions every time.

**Expected**: Query real card transaction history. If card provider integration isn't ready, at least return `[]` instead of fake transactions.

---

### 8. `POST /api/v1/relayer/submit` — random hash, always `pending`

**Current**: Generates a random hash and returns `{ status: "pending" }` without actually submitting to any bundler.

**Expected**: Submit the UserOperation to the Pimlico bundler (or configured ERC-4337 bundler) and return the real operation hash.

**Note**: `POST /api/v1/relayer/sponsor` already uses the Pimlico bundler service correctly — this endpoint should follow the same pattern for submission.

---

### 9. `GET /api/v1/relayer/userop/{hash}` — always returns `pending`

**Current**: Always returns `{ status: "pending", tx_hash: null }`.

**Expected**: Query the bundler for actual UserOperation status. Return `confirmed` with real `tx_hash` once included in a block.

---

### 10. `GET /api/v1/commerce/merchants` + `GET /api/v1/commerce/merchants/{id}` — hardcoded demo merchants

**Current**: Returns a static list of demo merchants (Coffee Shop, Book Store, etc.).

**Expected**: Query from `merchants` table. If merchant onboarding isn't ready yet, this can stay as-is for launch with a note in the app that merchant discovery is "coming soon".

**Priority**: Lowest — merchant payments via QR scan work regardless (QR contains merchant ID directly).

---

## P3 — Infrastructure / Config

### 11. SSL Certificate Pin Hashes

Mobile has a network security config at `plugins/withNetworkSecurity.js` with placeholder pin hashes:

```xml
<pin digest="SHA-256">BASE64_ENCODED_SPKI_SHA256_PRIMARY</pin>
<pin digest="SHA-256">BASE64_ENCODED_SPKI_SHA256_BACKUP</pin>
```

**Need from backend**: SPKI SHA-256 hashes for the production API SSL certificate and one backup certificate.

Generate with:
```bash
openssl s_client -connect api.zelta.app:443 | \
  openssl x509 -pubkey -noout | \
  openssl pkey -pubin -outform DER | \
  openssl dgst -sha256 -binary | base64
```

---

### 12. Production Environment Variables

Mobile needs these values for production builds:

| Variable | Current (dev) | Need |
|----------|--------------|------|
| `EXPO_PUBLIC_API_URL` | `http://finaegis.local` | Production HTTPS URL |
| `EXPO_PUBLIC_PUSHER_APP_KEY` | `9f58e8fb6a9be1aeed0e` | Production Pusher/Soketi key |
| `EXPO_PUBLIC_PIMLICO_API_KEY` | `pim_MMAzf2QoTy8qKAosp1XbT9` | Production Pimlico key |
| `EXPO_PUBLIC_WALLETCONNECT_PROJECT_ID` | `fafaedbb1813e29ce55181630e83b8cc` | Production WalletConnect ID |

---

### 13. Sanctions/Compliance Endpoint (future)

Mobile has stub functions `isSanctionedAddress()` and `isContractAddress()` that always return `false`. For compliance:

**Option A**: Backend provides `GET /api/v1/compliance/check-address?address=0x...` returning `{ sanctioned: bool, is_contract: bool }`.

**Option B**: Mobile integrates a client-side OFAC list (larger bundle, stale data).

Recommend Option A. Not a launch blocker if the app is geo-restricted initially, but required before opening to regulated markets.

---

## Summary by Priority

| Priority | Items | Effort |
|----------|-------|--------|
| **P0** | #1 recovery backup, #2 broadcasting auth, #3 privacy calldata | #2 is one-line fix; #1 and #3 need new controllers |
| **P1** | #4 recipient name field | Small change to existing query |
| **P2** | #5-#10 stub implementations | Varies — #8/#9 need bundler integration, rest are DB queries |
| **P3** | #11 SSL pins, #12 env vars, #13 sanctions | Config/infra |

---

*Mobile will separately fix all 23 path/method mismatches to align with existing backend routes. No backend changes needed for those.*
