# FinAegis v2.0.0 - Multi-Tenancy Architecture

> **Status**: ✅ COMPLETE (All Phases Merged)
> **Merged PRs**: #328, #330, #331, #332, #333, #334, #335, #336, #337
> **Completion Date**: January 2026
> **Branch**: `main`

---

## Implementation Status

| Phase | Description | PR | Status |
|-------|-------------|-----|--------|
| 1 | Foundation & POC | #328 | ✅ Complete |
| 2 | Migration Infrastructure | #337 | ✅ Complete |
| 3 | Event Sourcing Integration | #330 | ✅ Complete |
| 4 | Model Scoping | #331 | ✅ Complete |
| 5 | Queue Job Tenant Context | #332 | ✅ Complete |
| 6 | WebSocket Channel Auth | #333 | ✅ Complete |
| 7 | Filament Admin Filtering | #334 | ✅ Complete |
| 8 | Data Migration Tooling | #335 | ✅ Complete |
| 9 | Security Audit | #336 | ✅ Complete |

### Implementation Highlights

**Phase 1: Foundation**
- stancl/tenancy v3.9 integration
- Custom `Tenant` model with Team relationship
- `UsesTenantConnection` trait
- `InitializeTenancyByTeam` middleware
- `TeamTenantResolver` with caching

**Phase 2: Migration Infrastructure**
- 14 tenant migration files covering all domains
- Domains: Account, Banking, Lending, Compliance, Exchange, Stablecoin, Wallet, Treasury, CGO, Agent Protocol

**Phase 3-4: Model Integration**
- 83 models with `UsesTenantConnection` trait
- 16 event sourcing models with tenant awareness
- 5 snapshot models with tenant support

**Phase 5-6: Infrastructure**
- `TenantAwareJob` trait for queue jobs
- `TenantBroadcastEvent` trait for WebSocket
- `TenantChannelAuthorizer` for channel authorization

**Phase 7: Admin Panel**
- `TenantAwareResource` trait for Filament
- `TenantSelectorWidget` for tenant switching
- `FilamentTenantMiddleware` for admin context

**Phase 8: Data Migration**
- `TenantDataMigrationService` for data migration
- Artisan commands: `tenants:migrate-data`, `tenants:export-data`, `tenants:import-data`
- Support for JSON, CSV, SQL formats

**Phase 9: Security**
- 26 security tests for tenant isolation
- Cross-tenant access prevention tests
- Security audit documentation

---

## Executive Summary

This document outlines the architecture for implementing multi-tenancy in FinAegis v2.0.0. The recommended approach uses **stancl/tenancy with multi-database isolation** to meet financial compliance requirements while maintaining the existing Spatie Event Sourcing architecture.

### Key Decisions

| Decision | Choice | Rationale |
|----------|--------|-----------|
| Package | stancl/tenancy v3.x | Most comprehensive, best event sourcing support |
| Isolation Strategy | Multi-database | Compliance requirements, audit trails, data segregation |
| Tenant Identifier | Team-based (existing Jetstream teams) | Minimal migration, already in codebase |
| Event Sourcing | Per-tenant databases | Complete isolation of financial event streams |

---

## Architecture Overview

```
┌─────────────────────────────────────────────────────────────────────────────┐
│                         FinAegis v2.0.0 Architecture                         │
├─────────────────────────────────────────────────────────────────────────────┤
│                                                                             │
│  ┌─────────────────────────────────────────────────────────────────────┐   │
│  │                      CENTRAL DATABASE (Landlord)                     │   │
│  │  ┌──────────┐  ┌──────────┐  ┌──────────┐  ┌──────────────────┐   │   │
│  │  │ tenants  │  │  users   │  │  teams   │  │ system_settings  │   │   │
│  │  └──────────┘  └──────────┘  └──────────┘  └──────────────────┘   │   │
│  │  ┌──────────┐  ┌──────────┐  ┌──────────┐  ┌──────────────────┐   │   │
│  │  │  roles   │  │permissions│  │  assets  │  │ exchange_rates   │   │   │
│  │  └──────────┘  └──────────┘  └──────────┘  └──────────────────┘   │   │
│  └─────────────────────────────────────────────────────────────────────┘   │
│                                    │                                        │
│                    ┌───────────────┼───────────────┐                       │
│                    ▼               ▼               ▼                        │
│  ┌─────────────────────┐ ┌─────────────────────┐ ┌─────────────────────┐  │
│  │  TENANT DB: Acme    │ │  TENANT DB: Beta    │ │  TENANT DB: Corp    │  │
│  │                     │ │                     │ │                     │  │
│  │  ┌───────────────┐  │ │  ┌───────────────┐  │ │  ┌───────────────┐  │  │
│  │  │   accounts    │  │ │  │   accounts    │  │ │  │   accounts    │  │  │
│  │  │  transactions │  │ │  │  transactions │  │ │  │  transactions │  │  │
│  │  │   ledgers     │  │ │  │   ledgers     │  │ │  │   ledgers     │  │  │
│  │  └───────────────┘  │ │  └───────────────┘  │ │  └───────────────┘  │  │
│  │                     │ │                     │ │                     │  │
│  │  Event Sourcing:    │ │  Event Sourcing:    │ │  Event Sourcing:    │  │
│  │  ┌───────────────┐  │ │  ┌───────────────┐  │ │  ┌───────────────┐  │  │
│  │  │treasury_events│  │ │  │treasury_events│  │ │  │treasury_events│  │  │
│  │  │compliance_evts│  │ │  │compliance_evts│  │ │  │compliance_evts│  │  │
│  │  │exchange_events│  │ │  │exchange_events│  │ │  │exchange_events│  │  │
│  │  │  ...41 tables │  │ │  │  ...41 tables │  │ │  │  ...41 tables │  │  │
│  │  └───────────────┘  │ │  └───────────────┘  │ │  └───────────────┘  │  │
│  └─────────────────────┘ └─────────────────────┘ └─────────────────────┘  │
│                                                                             │
└─────────────────────────────────────────────────────────────────────────────┘
```

---

## Scope Analysis

### Impact Summary

| Category | Count | Effort |
|----------|-------|--------|
| Eloquent Models | 83 | HIGH |
| Event Sourcing Tables | 41 | CRITICAL |
| HTTP Controllers | 168 | MEDIUM |
| Domain Services | 170 | MEDIUM |
| Database Migrations | 204 | HIGH |
| Filament Resources | 141 | MEDIUM |
| Cache References | 564 | MEDIUM |
| Queue Jobs | 11+ | LOW |

### Table Classification

**CENTRAL DATABASE (Landlord) - ~25 tables**
- `tenants` - Tenant registry (NEW)
- `users` - User authentication
- `teams` - Team/organization data
- `roles`, `permissions` - Spatie Permission
- `assets` - Asset definitions (USD, BTC, etc.)
- `exchange_rates` - Global exchange rates
- `settings` - System configuration
- OAuth tables (Passport)
- Framework tables (jobs, cache, sessions)

**TENANT DATABASES - ~150 tables per tenant**
- `accounts`, `account_balances` - Core accounts
- `transactions`, `transfers`, `ledgers` - Financial records
- `bank_accounts`, `custodian_accounts` - Banking integration
- `kyc_verifications`, `compliance_alerts` - Compliance
- `loans`, `loan_applications` - Lending
- All 41 event sourcing tables
- All domain-specific data

---

## Implementation Phases

### Phase 1: Foundation (Week 1-2)

**1.1 Package Installation & Configuration**

```bash
composer require stancl/tenancy
php artisan tenancy:install
```

**1.2 Tenant Model Setup**

```php
// app/Models/Tenant.php
use Stancl\Tenancy\Database\Models\Tenant as BaseTenant;
use Stancl\Tenancy\Contracts\TenantWithDatabase;
use Stancl\Tenancy\Database\Concerns\HasDatabase;
use Stancl\Tenancy\Database\Concerns\HasDomains;

class Tenant extends BaseTenant implements TenantWithDatabase
{
    use HasDatabase, HasDomains;

    public static function getCustomColumns(): array
    {
        return [
            'id',
            'team_id',        // Link to existing teams
            'name',
            'plan',
            'trial_ends_at',
            'data',
        ];
    }

    public function team(): BelongsTo
    {
        return $this->belongsTo(Team::class);
    }
}
```

**1.3 Tenancy Configuration**

```php
// config/tenancy.php
return [
    'tenant_model' => \App\Models\Tenant::class,

    'central_domains' => [
        'finaegis.com',
        'admin.finaegis.com',
    ],

    'bootstrappers' => [
        Stancl\Tenancy\Bootstrappers\DatabaseTenancyBootstrapper::class,
        Stancl\Tenancy\Bootstrappers\CacheTenancyBootstrapper::class,
        Stancl\Tenancy\Bootstrappers\FilesystemTenancyBootstrapper::class,
        Stancl\Tenancy\Bootstrappers\QueueTenancyBootstrapper::class,
    ],

    'database' => [
        'central_connection' => env('DB_CONNECTION', 'mysql'),
        'template_tenant_connection' => 'tenant',
        'prefix' => 'tenant_',
        'suffix' => '',
    ],
];
```

**1.4 Database Connections**

```php
// config/database.php
'connections' => [
    'central' => [
        'driver' => 'mysql',
        'database' => env('DB_DATABASE', 'finaegis_central'),
        // ... standard config
    ],

    'tenant' => [
        'driver' => 'mysql',
        'database' => null, // Set dynamically
        // ... standard config
    ],
],
```

---

### Phase 2: Migration Infrastructure (Week 2-3)

**2.1 Central Migrations**

Move to `database/migrations/central/`:
- `create_users_table.php`
- `create_teams_table.php`
- `create_roles_tables.php` (Spatie)
- `create_assets_table.php`
- `create_exchange_rates_table.php`
- `create_tenants_table.php` (NEW)
- `create_domains_table.php` (NEW)

**2.2 Tenant Migrations**

Move to `database/migrations/tenant/`:
- All account-related migrations
- All transaction/transfer migrations
- All compliance migrations
- All event sourcing migrations (41 tables)
- All domain-specific migrations

**2.3 Event Sourcing Migration Updates**

```php
// database/migrations/tenant/create_treasury_events_table.php
Schema::create('treasury_events', function (Blueprint $table) {
    $table->id();
    $table->uuid('aggregate_uuid')->nullable()->index();
    $table->unsignedInteger('aggregate_version')->nullable();
    $table->integer('event_version')->default(1);
    $table->string('event_class');
    $table->jsonb('event_properties');
    $table->jsonb('meta_data');
    $table->timestamp('created_at');

    // Composite index for performance
    $table->index(['aggregate_uuid', 'aggregate_version']);
});
```

---

### Phase 3: Event Sourcing Integration (Week 3-4)

**3.1 Tenant-Aware Event Models**

```php
// app/Domain/Treasury/Models/TreasuryEvent.php
namespace App\Domain\Treasury\Models;

use Spatie\EventSourcing\StoredEvents\Models\EloquentStoredEvent;

class TreasuryEvent extends EloquentStoredEvent
{
    protected $table = 'treasury_events';
    protected $connection = 'tenant';
}
```

**3.2 Tenant-Aware Repositories**

```php
// app/Domain/Treasury/Repositories/TreasuryEventRepository.php
namespace App\Domain\Treasury\Repositories;

use Spatie\EventSourcing\StoredEvents\Repositories\EloquentStoredEventRepository;

class TreasuryEventRepository extends EloquentStoredEventRepository
{
    protected string $storedEventModel = TreasuryEvent::class;
}
```

**3.3 Aggregate Configuration**

```php
// app/Domain/Treasury/Aggregates/PortfolioAggregate.php
class PortfolioAggregate extends AggregateRoot
{
    protected function getStoredEventRepository(): StoredEventRepository
    {
        return app(TreasuryEventRepository::class);
    }

    protected function getSnapshotRepository(): SnapshotRepository
    {
        return app(TreasurySnapshotRepository::class);
    }
}
```

---

### Phase 4: Model Scoping (Week 4-5)

**4.1 UsesTenantConnection Trait**

```php
// app/Domain/Shared/Traits/UsesTenantConnection.php
namespace App\Domain\Shared\Traits;

trait UsesTenantConnection
{
    public function getConnectionName()
    {
        return 'tenant';
    }
}
```

**4.2 Apply to All Tenant Models**

```php
// app/Models/Account.php
class Account extends Model
{
    use UsesTenantConnection;

    // ... existing code
}
```

**4.3 Affected Models (83 total)**

Priority order:
1. `Account`, `AccountBalance`, `Transaction`, `Transfer`, `Ledger`
2. `BankAccount`, `CustodianAccount`, `PaymentDeposit`, `PaymentWithdrawal`
3. `KycVerification`, `ComplianceAlert`, `AmlScreening`, `FraudScore`
4. `Loan`, `LoanApplication`, `LoanCollateral`, `LoanRepayment`
5. All remaining domain models

---

### Phase 5: Service Layer Updates (Week 5-6)

**5.1 Tenant Context Service**

```php
// app/Services/TenantContextService.php
namespace App\Services;

use App\Models\Tenant;

class TenantContextService
{
    public function current(): ?Tenant
    {
        return tenant();
    }

    public function runForTenant(Tenant $tenant, callable $callback): mixed
    {
        return tenancy()->run($tenant, $callback);
    }

    public function assertTenantContext(): void
    {
        if (!tenant()) {
            throw new TenantContextException('No tenant context available');
        }
    }
}
```

**5.2 Update Domain Services**

```php
// Example: app/Domain/Account/Services/AccountService.php
class AccountService
{
    public function __construct(
        private TenantContextService $tenantContext,
        private AccountRepository $repository
    ) {}

    public function createAccount(CreateAccountDTO $dto): Account
    {
        $this->tenantContext->assertTenantContext();

        // Account will automatically use tenant connection
        return $this->repository->create($dto);
    }
}
```

---

### Phase 6: Cache & Queue Isolation (Week 6)

**6.1 Cache Bootstrapper (Automatic)**

With `CacheTenancyBootstrapper` enabled:
- Cache keys automatically prefixed with `tenant_{id}_`
- No code changes needed for standard Cache:: usage

**6.2 Queue Tenant Context**

```php
// app/Jobs/ProcessTransaction.php
class ProcessTransaction implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    // Tenant ID automatically serialized by QueueTenancyBootstrapper

    public function handle(): void
    {
        // Tenant context automatically restored
        $account = Account::find($this->accountId);
        // ...
    }
}
```

---

### Phase 7: Authentication & Authorization (Week 6-7)

**7.1 Tenant Identification Middleware**

```php
// app/Http/Middleware/InitializeTenancyByTeam.php
namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;

class InitializeTenancyByTeam
{
    public function handle(Request $request, Closure $next)
    {
        $user = $request->user();

        if ($user && $user->currentTeam) {
            $tenant = Tenant::where('team_id', $user->currentTeam->id)->first();

            if ($tenant) {
                tenancy()->initialize($tenant);
            }
        }

        return $next($request);
    }
}
```

**7.2 Route Configuration**

```php
// routes/web.php
Route::middleware([
    'auth:sanctum',
    'verified',
    InitializeTenancyByTeam::class,
])->group(function () {
    // All tenant-scoped routes
});

// routes/api.php
Route::middleware([
    'auth:api',
    InitializeTenancyByTeam::class,
])->group(function () {
    // All tenant-scoped API routes
});
```

---

### Phase 8: Filament Admin Updates (Week 7)

**8.1 Tenant-Aware Resources**

```php
// app/Filament/Admin/Resources/AccountResource.php
class AccountResource extends Resource
{
    public static function getEloquentQuery(): Builder
    {
        // Automatic via tenant connection
        return parent::getEloquentQuery();
    }
}
```

**8.2 Super Admin Panel (Central)**

```php
// For system-wide administration
// Separate Filament panel on central database
```

---

### Phase 9: Data Migration (Week 7-8)

**9.1 Migration Script**

```php
// app/Console/Commands/MigrateTenantsData.php
class MigrateTenantsData extends Command
{
    protected $signature = 'tenants:migrate-data {--team=}';

    public function handle()
    {
        $teams = $this->option('team')
            ? Team::where('id', $this->option('team'))->get()
            : Team::all();

        foreach ($teams as $team) {
            $this->info("Migrating team: {$team->name}");

            // Create tenant
            $tenant = Tenant::create([
                'team_id' => $team->id,
                'name' => $team->name,
            ]);

            // Run tenant migrations
            $tenant->run(function () {
                Artisan::call('migrate', [
                    '--path' => 'database/migrations/tenant',
                    '--force' => true,
                ]);
            });

            // Copy data
            $this->migrateTeamData($team, $tenant);
        }
    }

    private function migrateTeamData(Team $team, Tenant $tenant): void
    {
        $tenant->run(function () use ($team) {
            // Migrate accounts
            $accounts = DB::connection('central')
                ->table('accounts')
                ->where('team_id', $team->id)
                ->get();

            foreach ($accounts as $account) {
                DB::table('accounts')->insert((array) $account);
            }

            // Migrate event streams
            // ... similar for all tables
        });
    }
}
```

---

## Testing Strategy

### Test Categories

| Category | Count | Focus |
|----------|-------|-------|
| Unit Tests | 150+ | Model scoping, service isolation |
| Feature Tests | 200+ | Controller isolation, API access |
| Security Tests | 50+ | Data leakage prevention |
| Integration Tests | 30+ | Cross-tenant scenarios |

### Critical Test Scenarios

```php
// tests/Feature/MultiTenancy/TenantIsolationTest.php
class TenantIsolationTest extends TestCase
{
    public function test_tenant_cannot_access_other_tenant_accounts(): void
    {
        $tenant1 = Tenant::factory()->create();
        $tenant2 = Tenant::factory()->create();

        $account1 = tenancy()->run($tenant1, fn() =>
            Account::factory()->create()
        );

        tenancy()->run($tenant2, function () use ($account1) {
            $this->assertNull(Account::find($account1->id));
        });
    }

    public function test_event_sourcing_isolated_per_tenant(): void
    {
        $tenant1 = Tenant::factory()->create();
        $tenant2 = Tenant::factory()->create();

        // Create aggregate in tenant 1
        $aggregate = tenancy()->run($tenant1, fn() =>
            PortfolioAggregate::create()
                ->addAsset('USD', 1000)
                ->persist()
        );

        // Verify not visible in tenant 2
        tenancy()->run($tenant2, function () use ($aggregate) {
            $events = TreasuryEvent::where('aggregate_uuid', $aggregate->uuid())->get();
            $this->assertCount(0, $events);
        });
    }
}
```

---

## Risk Mitigation

### Critical Risks

| Risk | Impact | Mitigation |
|------|--------|------------|
| Data leakage between tenants | CRITICAL | Comprehensive testing, security audit |
| Event sourcing corruption | HIGH | Per-tenant databases, replay testing |
| Performance degradation | MEDIUM | Connection pooling, query optimization |
| Migration data loss | HIGH | Backup strategy, incremental migration |

### Security Checklist

- [ ] All models use tenant connection
- [ ] No raw queries without tenant context
- [ ] Cache keys properly prefixed
- [ ] Queue jobs restore tenant context
- [ ] API tokens scoped to tenant
- [ ] Filament resources filtered
- [ ] Event streams isolated
- [ ] File storage separated

---

## Success Metrics

| Metric | Target |
|--------|--------|
| Tenant Isolation | 100% - No cross-tenant data access |
| Event Sourcing Integrity | All aggregates replay correctly |
| Performance | < 10% overhead vs single-tenant |
| Migration Success | Zero data loss |
| Test Coverage | 90%+ for tenancy code |

---

## Timeline Summary

| Week | Phase | Deliverables |
|------|-------|--------------|
| 1-2 | Foundation | Package setup, config, tenant model |
| 2-3 | Migrations | Central/tenant migration split |
| 3-4 | Event Sourcing | Tenant-aware aggregates, repositories |
| 4-5 | Models | 83 models with tenant connection |
| 5-6 | Services | 170 services updated |
| 6 | Cache & Queue | Isolation verification |
| 6-7 | Auth | Tenant identification middleware |
| 7 | Filament | Admin panel updates |
| 7-8 | Data Migration | Production data migration |
| 8 | Testing | Comprehensive test suite |

---

## Next Steps

1. **Approval** - Review and approve this architecture plan
2. **Spike** - Create proof-of-concept with 2-3 domains
3. **Feature Branch** - Create `feature/v2.0.0-multi-tenancy`
4. **Incremental PRs** - Phase-by-phase implementation
5. **Security Audit** - External review before production

---

## References

- [stancl/tenancy Documentation](https://tenancyforlaravel.com/)
- [Spatie Event Sourcing](https://spatie.be/docs/laravel-event-sourcing)
- [Laravel Multi-Tenancy Best Practices](https://laravel.com/docs/multi-tenancy)
- [FinAegis VERSION_ROADMAP](./VERSION_ROADMAP.md)

---

*Document created: January 27, 2026*
*Last updated: January 27, 2026*
