<?php

declare(strict_types=1);

namespace Tests\Feature\Api\Partner;

use App\Domain\FinancialInstitution\Models\FinancialInstitutionApplication;
use App\Domain\FinancialInstitution\Models\FinancialInstitutionPartner;
use App\Domain\FinancialInstitution\Models\PartnerIntegration;
use Illuminate\Support\Str;
use Tests\TestCase;

class PartnerMarketplaceControllerTest extends TestCase
{
    private FinancialInstitutionPartner $partner;

    private string $clientSecret = 'test_secret_123';

    protected function setUp(): void
    {
        parent::setUp();
        $this->partner = $this->createPartner();
    }

    private function createPartnerApplication(): FinancialInstitutionApplication
    {
        return FinancialInstitutionApplication::create([
            'application_number'       => 'FIA-2026-' . fake()->unique()->numerify('#####'),
            'institution_name'         => 'Test Partner',
            'legal_name'               => 'Test Partner Ltd',
            'registration_number'      => 'REG-123456',
            'tax_id'                   => 'TAX-123456',
            'country'                  => 'US',
            'institution_type'         => 'fintech',
            'years_in_operation'       => 5,
            'contact_name'             => 'John Doe',
            'contact_email'            => 'john@test.com',
            'contact_phone'            => '+1234567890',
            'contact_position'         => 'CTO',
            'headquarters_address'     => '123 Test St',
            'headquarters_city'        => 'New York',
            'headquarters_postal_code' => '10001',
            'headquarters_country'     => 'US',
            'business_description'     => 'Test fintech partner',
            'target_markets'           => ['US', 'EU'],
            'product_offerings'        => ['payments'],
            'required_currencies'      => ['USD'],
            'integration_requirements' => ['api'],
            'status'                   => 'approved',
        ]);
    }

    private function createPartner(array $attributes = []): FinancialInstitutionPartner
    {
        $application = $this->createPartnerApplication();

        return FinancialInstitutionPartner::create(array_merge([
            'application_id'        => $application->id,
            'partner_code'          => 'TST-' . fake()->unique()->numerify('####'),
            'institution_name'      => 'Test Partner',
            'legal_name'            => 'Test Partner Ltd',
            'institution_type'      => 'fintech',
            'country'               => 'US',
            'status'                => 'active',
            'tier'                  => 'growth',
            'billing_cycle'         => 'monthly',
            'api_client_id'         => 'test_client_abc',
            'api_client_secret'     => encrypt($this->clientSecret),
            'webhook_secret'        => encrypt('webhook_secret_123'),
            'sandbox_enabled'       => true,
            'production_enabled'    => false,
            'rate_limit_per_minute' => 300,
            'fee_structure'         => ['base' => 0],
            'risk_rating'           => 'low',
            'risk_score'            => 10.00,
            'primary_contact'       => ['name' => 'Test', 'email' => 'test@example.com'],
        ], $attributes));
    }

    private function partnerHeaders(): array
    {
        return [
            'X-Partner-Client-Id'     => 'test_client_abc',
            'X-Partner-Client-Secret' => $this->clientSecret,
        ];
    }

    private function createIntegration(array $attributes = []): PartnerIntegration
    {
        return PartnerIntegration::create(array_merge([
            'uuid'       => (string) Str::uuid(),
            'partner_id' => $this->partner->id,
            'category'   => 'payment_processors',
            'provider'   => 'stripe',
            'status'     => 'active',
            'config'     => ['api_key' => 'test'],
            'metadata'   => [],
        ], $attributes));
    }

    public function test_list_available_integrations(): void
    {
        $response = $this->getJson('/api/partner/v1/marketplace', $this->partnerHeaders());

        $response->assertOk()
            ->assertJsonPath('success', true)
            ->assertJsonStructure(['success', 'data' => ['payment_processors', 'analytics']]);
    }

    public function test_get_partner_integrations(): void
    {
        $this->createIntegration();

        $response = $this->getJson('/api/partner/v1/marketplace/integrations', $this->partnerHeaders());

        $response->assertOk()
            ->assertJsonPath('success', true)
            ->assertJsonCount(1, 'data');
    }

    public function test_enable_integration(): void
    {
        $response = $this->postJson('/api/partner/v1/marketplace/integrations', [
            'category' => 'payment_processors',
            'provider' => 'stripe',
            'config'   => ['api_key' => 'sk_test_123'],
        ], $this->partnerHeaders());

        $response->assertStatus(201)
            ->assertJsonPath('success', true);
    }

    public function test_enable_integration_invalid_category(): void
    {
        $response = $this->postJson('/api/partner/v1/marketplace/integrations', [
            'category' => 'nonexistent',
            'provider' => 'stripe',
        ], $this->partnerHeaders());

        $response->assertStatus(422);
    }

    public function test_enable_integration_missing_fields(): void
    {
        $response = $this->postJson('/api/partner/v1/marketplace/integrations', [], $this->partnerHeaders());

        $response->assertStatus(422);
    }

    public function test_disable_integration(): void
    {
        $integration = $this->createIntegration();

        $response = $this->deleteJson(
            "/api/partner/v1/marketplace/integrations/{$integration->id}",
            [],
            $this->partnerHeaders()
        );

        $response->assertOk()
            ->assertJsonPath('success', true);
    }

    public function test_disable_integration_not_found(): void
    {
        $response = $this->deleteJson(
            '/api/partner/v1/marketplace/integrations/999999',
            [],
            $this->partnerHeaders()
        );

        $response->assertStatus(404);
    }

    public function test_test_connection(): void
    {
        $integration = $this->createIntegration();

        $response = $this->postJson(
            "/api/partner/v1/marketplace/integrations/{$integration->id}/test",
            [],
            $this->partnerHeaders()
        );

        $response->assertOk()
            ->assertJsonPath('success', true)
            ->assertJsonStructure(['success', 'data' => ['latency_ms']]);
    }

    public function test_integration_health(): void
    {
        $this->createIntegration();

        $response = $this->getJson('/api/partner/v1/marketplace/health', $this->partnerHeaders());

        $response->assertOk()
            ->assertJsonPath('success', true)
            ->assertJsonPath('data.total', 1)
            ->assertJsonPath('data.active', 1)
            ->assertJsonPath('data.health_score', 100);
    }
}
