<?php

declare(strict_types=1);

use Tests\TestCase;

uses(TestCase::class);

use App\Domain\MobilePayment\Enums\PaymentAsset;
use App\Domain\MobilePayment\Enums\PaymentNetwork;
use App\Domain\MobilePayment\Services\ReceiveAddressService;

describe('ReceiveAddressService', function (): void {
    it('can be instantiated', function (): void {
        $service = new ReceiveAddressService();

        expect($service)->toBeInstanceOf(ReceiveAddressService::class);
    });

    it('has getReceiveAddress method with correct parameters', function (): void {
        $service = new ReceiveAddressService();
        $reflection = new ReflectionClass($service);
        $method = $reflection->getMethod('getReceiveAddress');

        expect($method->getNumberOfParameters())->toBe(3);

        $params = $method->getParameters();
        expect($params[0]->getName())->toBe('userId');
        expect($params[1]->getName())->toBe('network');
        expect($params[2]->getName())->toBe('asset');
    });
});

describe('ReceiveAddressService QR Value', function (): void {
    it('builds Solana QR value via reflection', function (): void {
        $service = new ReceiveAddressService();
        $reflection = new ReflectionClass($service);
        $method = $reflection->getMethod('buildQrValue');
        $method->setAccessible(true);

        $qr = $method->invoke($service, 'testAddr123', PaymentNetwork::SOLANA, PaymentAsset::USDC);

        expect($qr)->toBe('solana:testAddr123?spl-token=USDC');
    });

    it('builds Tron QR value as plain address', function (): void {
        $service = new ReceiveAddressService();
        $reflection = new ReflectionClass($service);
        $method = $reflection->getMethod('buildQrValue');
        $method->setAccessible(true);

        $qr = $method->invoke($service, 'TABCDEF1234567890123456789012345', PaymentNetwork::TRON, PaymentAsset::USDC);

        expect($qr)->toBe('TABCDEF1234567890123456789012345');
    });
});

describe('ReceiveAddressService Demo Address Generation', function (): void {
    it('generates base58-like addresses for Solana via reflection', function (): void {
        $service = new ReceiveAddressService();
        $reflection = new ReflectionClass($service);
        $method = $reflection->getMethod('generateDemoAddress');
        $method->setAccessible(true);

        $address = $method->invoke($service, 1, PaymentNetwork::SOLANA);

        expect($address)->toBeString();
        expect(strlen($address))->toBe(44);
    });

    it('generates T-prefixed addresses for Tron via reflection', function (): void {
        $service = new ReceiveAddressService();
        $reflection = new ReflectionClass($service);
        $method = $reflection->getMethod('generateDemoAddress');
        $method->setAccessible(true);

        $address = $method->invoke($service, 1, PaymentNetwork::TRON);

        expect($address)->toStartWith('T');
        expect(strlen($address))->toBe(34);
    });

    it('generates deterministic addresses for same user and network', function (): void {
        $service = new ReceiveAddressService();
        $reflection = new ReflectionClass($service);
        $method = $reflection->getMethod('generateDemoAddress');
        $method->setAccessible(true);

        $addr1 = $method->invoke($service, 42, PaymentNetwork::SOLANA);
        $addr2 = $method->invoke($service, 42, PaymentNetwork::SOLANA);

        expect($addr1)->toBe($addr2);
    });

    it('generates different addresses for different users', function (): void {
        $service = new ReceiveAddressService();
        $reflection = new ReflectionClass($service);
        $method = $reflection->getMethod('generateDemoAddress');
        $method->setAccessible(true);

        $addr1 = $method->invoke($service, 1, PaymentNetwork::SOLANA);
        $addr2 = $method->invoke($service, 2, PaymentNetwork::SOLANA);

        expect($addr1)->not->toBe($addr2);
    });

    it('generates different addresses for different networks', function (): void {
        $service = new ReceiveAddressService();
        $reflection = new ReflectionClass($service);
        $method = $reflection->getMethod('generateDemoAddress');
        $method->setAccessible(true);

        $addr1 = $method->invoke($service, 1, PaymentNetwork::SOLANA);
        $addr2 = $method->invoke($service, 1, PaymentNetwork::TRON);

        expect($addr1)->not->toBe($addr2);
    });
});
