<?php

declare(strict_types=1);

namespace Tests\Unit\Domain\Monitoring\Services;

use App\Domain\Monitoring\Services\PrometheusExporter;
use App\Models\User;
// Note: Domain models commented out as they may not have factories
// use App\Domain\Account\Models\Account;
// use App\Domain\Asset\Models\Asset;
// use App\Domain\Account\Models\Transaction;
use Illuminate\Support\Facades\Cache;
use Tests\TestCase;

class PrometheusExporterTest extends TestCase
{
    private PrometheusExporter $exporter;

    protected function setUp(): void
    {
        parent::setUp();

        $this->exporter = app(PrometheusExporter::class);
    }

    public function test_exports_prometheus_format(): void
    {
        // Act
        $output = $this->exporter->export();

        // Assert
        // Already asserted as string by return type
        $this->assertNotEmpty($output);
        $this->assertStringContainsString('# HELP', $output);
        $this->assertStringContainsString('# TYPE', $output);
    }

    public function test_exports_application_metrics(): void
    {
        // Arrange - Create some test data
        User::factory()->count(5)->create();

        // Act
        $output = $this->exporter->export();

        // Assert
        $this->assertStringContainsString('app_users_total', $output);
        $this->assertStringContainsString('app_uptime_seconds', $output);
        $this->assertStringContainsString('app_memory_usage_bytes', $output);
        $this->assertStringContainsString('app_cache_', $output);
    }

    public function test_exports_business_metrics(): void
    {
        // Arrange - Create business data
        $user = User::factory()->create();
        // Note: Skipping domain models that may not have factories
        // Just testing with User model

        // Act
        $output = $this->exporter->export();

        // Assert
        $this->assertStringContainsString('app_users_total', $output);
        // Check that we have at least 1 user (use \d+ to match any number)
        $this->assertMatchesRegularExpression('/app_users_total\s+\d+/', $output);
    }

    public function test_exports_infrastructure_metrics(): void
    {
        // Act
        $output = $this->exporter->export();

        // Assert
        $this->assertStringContainsString('infra_db_connections', $output);
        $this->assertStringContainsString('infra_queue_size', $output);
        $this->assertStringContainsString('infra_redis_memory_bytes', $output);
    }

    public function test_exports_http_metrics(): void
    {
        // Arrange - Simulate some HTTP metrics
        Cache::put('metrics:http:requests:total', 1000);
        Cache::put('metrics:http:requests:status:200', 950);
        Cache::put('metrics:http:requests:status:500', 50);
        Cache::put('metrics:http:duration:average', 0.250);

        // Act
        $output = $this->exporter->export();

        // Assert
        $this->assertStringContainsString('http_requests_total', $output);
        $this->assertStringContainsString('http_request_duration_seconds', $output);
    }

    public function test_exports_cache_metrics(): void
    {
        // Arrange - Set some cache metrics
        Cache::put('test_key_1', 'value1');
        Cache::put('test_key_2', 'value2');
        Cache::put('metrics:cache:hits', 100);
        Cache::put('metrics:cache:misses', 10);

        // Act
        $output = $this->exporter->export();

        // Assert
        $this->assertStringContainsString('app_cache_hits_total', $output);
        $this->assertStringContainsString('app_cache_misses_total', $output);
    }

    public function test_exports_queue_metrics(): void
    {
        // Arrange
        Cache::put('metrics:queue:jobs', 50);
        Cache::put('metrics:queue:failed', 2);

        // Act
        $output = $this->exporter->export();

        // Assert
        $this->assertStringContainsString('infra_queue_size', $output);
        $this->assertStringContainsString('infra_queue_failed_total', $output);
    }

    public function test_exports_database_metrics(): void
    {
        // Act
        $output = $this->exporter->export();

        // Assert
        $this->assertStringContainsString('infra_db_connections', $output);
        $this->assertStringContainsString('infra_db_queries_total', $output);
    }

    public function test_exports_with_labels(): void
    {
        // Arrange
        // Set metrics with labels via cache
        Cache::put('metrics:http:requests:status:200', 10);
        Cache::put('metrics:http:requests:status:404', 2);
        Cache::put('metrics:http:requests:status:500', 1);

        // Act
        $output = $this->exporter->export();

        // Assert
        $this->assertStringContainsString('http_requests_total', $output);
        // Prometheus format uses labels in curly braces
        if (strpos($output, 'status=') !== false) {
            $this->assertMatchesRegularExpression('/\{[^}]*status=\"?\w+\"?[^}]*\}/', $output);
        }
    }

    public function test_handles_empty_metrics_gracefully(): void
    {
        // Act
        $output = $this->exporter->export();

        // Assert
        // Already asserted as string by return type
        $this->assertNotEmpty($output);
        // Should still have system metrics even with no business data
        $this->assertStringContainsString('app_uptime_seconds', $output);
        $this->assertStringContainsString('app_memory_usage_bytes', $output);
    }

    public function test_metric_names_follow_prometheus_convention(): void
    {
        // Act
        $output = $this->exporter->export();

        // Assert - Check naming conventions
        $lines = explode("\n", $output);
        foreach ($lines as $line) {
            if (strpos($line, '# HELP') === 0 || strpos($line, '# TYPE') === 0) {
                // Check metric names in HELP and TYPE lines
                if (preg_match('/# (?:HELP|TYPE) ([a-z_][a-z0-9_]*)/i', $line, $matches)) {
                    $metricName = $matches[1];
                    // Prometheus naming convention: lowercase with underscores
                    $this->assertMatchesRegularExpression('/^[a-z][a-z0-9_]*$/', $metricName);
                }
            }
        }
    }

    public function test_exports_workflow_metrics(): void
    {
        // Arrange
        Cache::put('metrics:workflows:started', 10);
        Cache::put('metrics:workflows:completed', 8);
        Cache::put('metrics:workflows:failed', 2);

        // Act
        $output = $this->exporter->export();

        // Assert
        $this->assertStringContainsString('workflow_executions_total', $output);
    }

    public function test_exports_event_metrics(): void
    {
        // Arrange
        Cache::put('metrics:events:processed', 1000);
        Cache::put('metrics:events:failed', 5);

        // Act
        $output = $this->exporter->export();

        // Assert
        $this->assertStringContainsString('events_processed_total', $output);
    }
}
